package oracle.spatial.georaster.samples;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.geom.AffineTransform;
import java.awt.image.RenderedImage;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;

import oracle.spatial.georaster.JGeoRaster;
import oracle.spatial.georaster.RasterInfo;


/**
 * This is a sample java class, which shows how to implement a simple GeoRaster object Viewer
 * based on GeoRaster Java API library.
 * <p>
 */
public class Viewer extends JPanel implements AdjustmentListener
{
    private ImagePanel m_imagePanel=new ImagePanel();
    JScrollPane m_scrollPane;
    JGeoRaster m_jgeor;
    int x=0;
    int y=0;
    
    /**
     * constructor
     */
    public Viewer()
    {
        super();
        setLayout(new GridBagLayout());
        m_scrollPane=new JScrollPane(m_imagePanel); 
        add(m_scrollPane,new GridBagConstraints(0,0,1,1,1.0D,1.0D,GridBagConstraints.NORTHWEST,GridBagConstraints.BOTH,new Insets(0,0,0,0),0,0));
        m_scrollPane.setVisible(true);
        m_scrollPane.getHorizontalScrollBar().setUnitIncrement(8);
        m_scrollPane.getVerticalScrollBar().setUnitIncrement(8);
        m_scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        m_scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);
        m_scrollPane.getHorizontalScrollBar().addAdjustmentListener(this);
        m_scrollPane.getVerticalScrollBar().addAdjustmentListener(this);
    }
    /**
     * set a Georaster object to display
     */
    public void setJGeoRaster(JGeoRaster jgeor)
    {
        m_jgeor=jgeor;
        long width=
            jgeor.getMetadataObject().getRasterInfo().getDimensionSize(RasterInfo.COLUMN_DIMENSION).longValue();
        long height=
            jgeor.getMetadataObject().getRasterInfo().getDimensionSize(RasterInfo.ROW_DIMENSION).longValue();
        m_imagePanel.setPreferredSize(new Dimension((int)width,(int)height));
        m_scrollPane.getHorizontalScrollBar().setValue(0);
        m_scrollPane.getVerticalScrollBar().setValue(0);
        m_imagePanel.fetchImage(m_jgeor,0,0);
        m_imagePanel.revalidate();
    }

    /**
     * 
     * this method deals with scroll bar change
     * 
     */
    public void adjustmentValueChanged(AdjustmentEvent e)
    {
        if((m_jgeor==null)||(e.getValueIsAdjusting())||
           ((m_scrollPane.getHorizontalScrollBar().getModel().getValue()==x)&&
            (m_scrollPane.getVerticalScrollBar().getModel().getValue()==y)))
        {
            return;
        }
        x=m_scrollPane.getHorizontalScrollBar().getModel().getValue();
        y=m_scrollPane.getVerticalScrollBar().getModel().getValue();
        m_imagePanel.fetchImage(m_jgeor,x,y);
    }

    /**
     * This inner class is a customized JPanel to display a GeoRaster image.
     * <p>
     */
    class ImagePanel extends JPanel
    {
        RenderedImage m_image=null;
        int m_oriPixX=0;
        int m_oriPixY=0;
        
        /* constructor */ 
        ImagePanel()
        {
            super();
            setBackground(new Color(50,50,50));
        }
        /* retrieve a GeoRaster image from server side.*/
        private boolean fetchImage(JGeoRaster jgeor,int oriX,int oriY)
        {
            try
            {
                m_image=jgeor.getGeoRasterImageObject().getRasterImage(
                                    0,
                                    oriY,
                                    oriX,
                                    (int)Math.min(oriY+600,
                                    jgeor.getMetadataObject().getRasterInfo().getDimensionSize(RasterInfo.ROW_DIMENSION).longValue()-1),
                                    (int)Math.min(oriX+800,
                                    jgeor.getMetadataObject().getRasterInfo().getDimensionSize(RasterInfo.COLUMN_DIMENSION).longValue()-1),new long[4]);

                if(m_image==null)
                    return false;
                m_oriPixX=oriX;
                m_oriPixY=oriY;
            }
            catch(Exception ee)
            {
                ee.printStackTrace();
            }
            repaint();
            return true;
        }

        /* draw a GeoRaster image on the current panel.*/
        public void paint(Graphics g)
        {
            try
            {
                super.paint(g);
                if(m_image!=null)
                {
                    AffineTransform af=AffineTransform.getTranslateInstance(m_oriPixX,m_oriPixY);
                    ((Graphics2D)g).drawRenderedImage(m_image,af);
                }
            }
            catch(Exception e)
            {
                e.printStackTrace();
            }
        }
    }
}


